/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xhost.c 
 *       Download and upload functions between host/file 
 *       and system/internal memory.
 * -----------------------------------------------------------------*/

#define UPLOAD_RETRY_HACK

#include <xutil.h>
#ifdef BEST_DEBUG
#define DBG_OUT(a)          BESTX_PRINTF a
#else
#define DBG_OUT(a)          (void) (0)
#endif
#include <xberror.h>
#include <xpci.h>

#include <xlasterr.h>
#include <xtypedef.h>
#include <xcmd.h>
#include <xaddrmap.h>

#include <xboard.h>
#include <xdynamic.h>
#include <xerrcapi.h>
#include <xhostdef.h>
#include <xhost.h>
#include <xiocommo.h>
#include <xri.h>
#include <xregcons.h>
#include <xsession.h>
#include <xexer.h>
#include <xdecode.h>

#include <sys/stat.h>

#ifndef BEST_FIRMWARE
#include <xidentos.h>
#endif

#define FAST_HOST_THRESHOLD 112

/* The define has to meet the following constraints:
   - < 127 for E2925
   - < 119 for padding
   - multiple of 8
   - as large as possible
   CZ
   */


/* *********************************************************************
 * host sysmem fill/dump: not in firmware
 * *********************************************************************/


#ifndef BEST_FIRMWARE

#define FOUR_X(a)      a, a, a, a
#define THIRTY_TWO_X(a) FOUR_X(FOUR_X(a)), FOUR_X(FOUR_X(a))

static bx_int32 sm_command[BX_MAXHANDLES] = 
{
  THIRTY_TWO_X(BX_RIBLK_BUSCMD_MEM_READDWORD)
};

static bx_int32 sm_bufsize[BX_MAXHANDLES] = 
{
  THIRTY_TWO_X(0)
};

static bx_int32 sm_verify[BX_MAXHANDLES] = 
{
  THIRTY_TWO_X(0)
};


#ifndef MIN
#define MIN(a,b) ( (a) < (b) ? (a) : (b) )
#endif

/* transfers any number of dwords using bytens given.
   IMPORTANT: buscommand must be setup in advance! */

static bx_errtype DoTransfer(bx_handletype handle,
    bx_int32 busaddress,
    bx_int32 buscmd,
    bx_int32 intaddress,
    bx_int32 nofbytes)
{
  BX_TRY_VARS;
    
  BX_TRY_BEGIN
  {
    /* bx_int32 status; */

    BX_TRY(BestXRIBlockDefaultSet(handle, 0));
    BX_TRY(BestXRIBlockSet(handle, 0, BX_RIBLK_BUSADDR_LO, busaddress));
    BX_TRY(BestXRIBlockSet(handle, 0, BX_RIBLK_RESOURCE, BX_RIBLK_RESOURCE_DATAMEM));
    BX_TRY(BestXRIBlockSet(handle, 0, BX_RIBLK_BUSCMD, buscmd));
    BX_TRY(BestXRIBlockSet(handle, 0, BX_RIBLK_INTADDR, intaddress));
    BX_TRY(BestXRIBlockSet(handle, 0, BX_RIBLK_NUMBYTES, nofbytes));
    /* this is handled in hardware
    BX_TRY(BestXRIBlockSet(handle, 0, BX_RIBLK_BYTEN, ((~((~(0xf<<nofbytes))<<(busaddress&0x3)))&0x0f)));
*/
    BX_TRY(BestXRIGenDefaultSet(handle)); /* set NUMBEH and NUMBLK to 1 */
    BX_TRY(BestXRIBehDefaultSet(handle, 0)); /* set default RIBehavior */
    BX_TRY(BestXRIBehSet(handle, 0, BX_RIBEH_REQ64, 0)); /* don't try 64 bit transfer */
    BX_TRY(BestXRIProg(handle));
    BX_TRY(BestXExerciserGenDefaultSet(handle));
    BX_TRY(BestXExerciserGenProg(handle));
    /* enable rt decoder */
    BX_TRY(BestXTDecoderSet(handle, BX_DEC_RT, BX_DECP_SIZE, 1));

    BX_TRY(BestXDirectRegWrite(handle, BX_REG_MSM_ABORT_REG, 2, 1)); /* resume on master abort ( dont know how to clear master abort ) */

    BX_TRY(BestXExerciserRun(handle));

#if 0   /* TBD: Better implementation */ 
    do  /* wait for completion */
    {
      BX_TRY(BestXStatusRead(handle, BX_STAT_TEST, &status));
      BX_TRY_FAIL(status & BX_STAT_TEST_IABORT ? BX_E_MASTER_ABORT : BX_E_OK);
      count++;
    } while ( (status & BX_STAT_TEST_RUNNING) );
#endif
  }

  BX_ERRETURN(BX_TRY_RET);

}

bx_errtype EXPORT BestXHostPCIRegWrite(
    bx_handletype handle,
    bx_addrspacetype addrspace,
    bx_int32 bus_addr,
    bx_int32 reg_value,
    bx_sizetype wordsize
)
{
  BX_DECLARE_FUNCNAME("BestXHostPCIRegWrite [hpregwrite]");

  BX_TRY_VARS;

  bx_int32 qwval[2];
  bx_int8 mbuf[8];
  bx_int32 intaddr;
  bx_int32 buscmd;
  bx_int32 offset;
  const bx_generic_infotype *mygen;    /* for datamem size */
  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI);
    
    BX_TRY_FCT_PARAM_RANGE(addrspace, BX_ADDRSPACE_CONFIG, BX_ADDRSPACE_MEM);
    BX_TRY_FCT_PARAM_RANGE(wordsize, BX_SIZE_BYTE, BX_SIZE_DWORD);
    BX_FCT_PARAM_CHK_R(bus_addr, (bus_addr&0x3) + wordsize > 4, "Address & Size crosses DWORD boundary");

    /* build dword */
    offset = bus_addr & 0x03;

    /* mask the bits that are not in wordsize */
    switch (wordsize)
    {
      case BX_SIZE_BYTE:
        reg_value &= 0x000000ff;
        break;

      case BX_SIZE_WORD:
        reg_value &= 0x0000ffff;
        break;
    }
    /* shift to the correct qword position */
    qwval[(bus_addr & 0x04)?1:0] = reg_value << (int) (offset << 3);

    /* create buscmd and bus_addr */
    switch (addrspace)
    {
      case BX_ADDRSPACE_CONFIG:
        /* CONFIG address space */
        bus_addr &= 0xfffffffcUL;
        /* check, if we have to set bit 0 of busaddr for config cycle type 1 */
        buscmd = BX_RIBLK_BUSCMD_CONFIG_WRITE;
        break;

      case BX_ADDRSPACE_IO:
        /* IO address space */
        buscmd = BX_RIBLK_BUSCMD_IO_WRITE;
        break;

      case BX_ADDRSPACE_MEM:
        /* MEM address space */
/*        bus_addr &= 0xfffffffcUL; */
        buscmd = BX_RIBLK_BUSCMD_MEM_WRITE;
        break;

      default:
        BX_TRY_ERROR(BX_E_INVALID_CASE);
    }

    BESTX_MEMCPY( (void *) mbuf, (void *) qwval, sizeof(mbuf) );

    /* intaddr = last qword address in data memory */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_DATA_MEM, &mygen));
    intaddr = mygen->depth - (mygen->num_blocks + 1);

    BX_TRY(BestXDataMemWrite(handle, intaddr, mygen->num_blocks + 1, &mbuf[0]));

    BX_TRY(DoTransfer(handle, bus_addr, buscmd, intaddr, wordsize));

  }
  
  BX_ERRETURN(BX_TRY_RET);
}

bx_errtype EXPORT BestXHostPCIRegRead(
    bx_handletype handle,
    bx_addrspacetype addrspace,
    bx_int32 bus_addr,
    bx_int32 * regvalue_ptr,
    bx_sizetype wordsize
)
{
  BX_DECLARE_FUNCNAME("BestXHostPCIRegRead [hpregread]");

  BX_TRY_VARS;

  bx_int32 qwval[2];
  bx_int8 mbuf[8];
  bx_int32 intaddr;
  bx_int32 buscmd;
  const bx_generic_infotype *mygen;    /* for datamem depth */
  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI);
    
    BX_TRY_FCT_PARAM_RANGE(addrspace, BX_ADDRSPACE_CONFIG, BX_ADDRSPACE_MEM);
    BX_TRY_FCT_PARAM_NULL_POINTER(regvalue_ptr);
    BX_TRY_FCT_PARAM_RANGE(wordsize, BX_SIZE_BYTE, BX_SIZE_DWORD);

    BX_FCT_PARAM_CHK_R(bus_addr, (bus_addr&0x3) + wordsize > 4, "Address & Size crosses DWORD boundary");

    /* create buscmd and modify bus_addr */
    switch (addrspace)
    {
      case BX_ADDRSPACE_CONFIG:
        /* CONFIG address space */
        bus_addr &= 0xfffffffcUL;
        buscmd = BX_RIBLK_BUSCMD_CONFIG_READ;
        break;

      case BX_ADDRSPACE_IO:
        /* IO address space */
        buscmd = BX_RIBLK_BUSCMD_IO_READ;
        break;

      case BX_ADDRSPACE_MEM:
        /* MEM address space */
        buscmd = BX_RIBLK_BUSCMD_MEM_READDWORD;
        break;

      default:
        BX_TRY_ERROR(BX_E_INVALID_CASE);
    }

    /* intaddr = last qword address of data memory */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_DATA_MEM, &mygen));
    intaddr = (mygen->depth - (mygen->num_blocks + 1));

    BX_TRY(DoTransfer(handle, bus_addr, buscmd, intaddr, wordsize));

    /* get the dword from internal memory */
    BX_TRY(BestXDataMemRead(handle, intaddr, mygen->num_blocks + 1, &mbuf[0]));

    BESTX_MEMCPY( (void *) qwval, (void *) mbuf, sizeof(qwval) );

    *regvalue_ptr = (qwval[(bus_addr & 0x04)?1:0] >> ((bus_addr&3) << 3));
    /* get correct value from dword */
    switch (wordsize)
    {
      case BX_SIZE_BYTE:
        *regvalue_ptr &= 0x000000ff;
        break;

      case BX_SIZE_WORD:
        *regvalue_ptr &= 0x0000ffff;
        break;

    }
  }

  BX_ERRETURN(BX_TRY_RET);
}
#endif

/* ********************************************************************
 * end of host functions
 * ********************************************************************/


#ifndef BEST_FIRMWARE

bx_errtype EXPORT BestXDataMemInit(bx_handletype handle)
{
  const bx_generic_infotype *mygen;
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI); 
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_DATA_MEM, &mygen));
    BX_TRY(BestXDataMemPatternFill(handle,0,mygen->depth,0,0,0,0,BX_PATT_CONST32));
  }
  BX_ERRETURN(BX_TRY_RET);
}
#endif

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDataMemReadFile(
 *
 * Purpose  : Writes datamemory to binary file
 *            For debugging purposes only !
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDataMemReadFile(bx_handletype handle, bx_int32 offset, bx_int32 numbytes, bx_charptrtype filename)
{
  BX_DECLARE_FUNCNAME("BestXDataMemReadFile [datamemreadfile]");
  
  bx_int8ptr data=NULL;
  const bx_generic_infotype *GenericDataMemInfo;
  FILE *fp=NULL;
  bx_int32 byteswritten=0;
  bx_int32 depth;
  char reason[256];

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI); 
    BX_FCT_PARAM_NULL_POINTER_CHK(filename);

    /* Check 128bit-Alignment */
    BX_FCT_PARAM_ALIGNMENT_CHK(offset,16);
    BX_FCT_PARAM_ALIGNMENT_CHK(numbytes,16);

    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_DATA_MEM, &GenericDataMemInfo));
    depth = GenericDataMemInfo->depth;

    BESTX_SPRINTF(reason, "greater than data memory (%ld)", depth);
    BX_TRY_FCT_PARAM_R(1, offset > depth, reason);

    BESTX_SPRINTF(reason, "exceeds data memory (%ld)", depth);
    BX_TRY_FCT_PARAM_R(2, offset + numbytes > depth, reason);

    
    if ((fp = BESTX_FOPEN(filename, "wb")) == (FILE *) NULL)
    {
      BX_TRY_FAIL(BX_E_FILE_OPEN);
    }

    data = (bx_int8ptr) BestXMemMalloc((size_t)numbytes);
    if (data==NULL)
    {
      BX_TRY_FAIL(BX_E_HOST_MEM_FULL);
    }    

    /* Read datamemory */
    BX_TRY(BestXDataMemRead(handle,offset,numbytes,data));
  
    /* Write to file */
    byteswritten=fwrite((void *)data,1,numbytes,fp);
    BestXFree(data);data=NULL;
    BESTX_FCLOSE(fp);fp=NULL;
    BX_E_ERROR_MSG_SET("BestXDataMemDump: Error while writing to file")
    BX_TRY_FAIL(byteswritten==numbytes?BX_E_OK:BX_E_ERROR);
  }
  BX_TRY_CATCH
  {
    if (data) BestXFree(data);
    if (fp) BESTX_FCLOSE(fp);
  }
  BX_ERRETURN(BX_TRY_RET);
}
/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDataMemWriteFile(
 *
 * Purpose  : Fills datamemory from binary file
 *            For debugging purposes only !
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDataMemWriteFile(bx_handletype handle, bx_int32 offset, bx_charptrtype filename)
{
  BX_DECLARE_FUNCNAME("BestXDataMemWriteFile [datamemwritefile]");
  
  bx_int8ptr data=NULL;
  const bx_generic_infotype *GenericDataMemInfo;
  FILE *fp=NULL;
  bx_int32 byteswritten=0;
  bx_int32 depth;
  char reason[256];
  struct stat st;

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI); 
    BX_FCT_PARAM_NULL_POINTER_CHK(filename);

    /* Check 128bit-Alignment */
    BX_FCT_PARAM_ALIGNMENT_CHK(offset,16);

    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_DATA_MEM, &GenericDataMemInfo));
    depth = GenericDataMemInfo->depth;

    if (stat(filename, &st))
    {
      return BX_E_FILE_OPEN;
    }

    BESTX_SPRINTF(reason, "exceeds data memory (%ld)", depth);
    BX_TRY_FCT_PARAM_R(2, offset + st.st_size > depth, reason);

    if ((fp = BESTX_FOPEN(filename, "rb")) == (FILE *) NULL)
    {
      BX_TRY_FAIL(BX_E_FILE_OPEN);
    }

    data = (bx_int8ptr) BestXMemMalloc((size_t)st.st_size);
    if (data==NULL)
    {
      BX_TRY_FAIL(BX_E_HOST_MEM_FULL);
    }    

    if (BESTX_FREAD(data,sizeof(char),st.st_size,fp)!=(size_t)st.st_size)
    {
      /* Should never happen */
      BX_E_ERROR_MSG_SET("Input file: Wrong size");
      BX_TRY_FAIL(BX_E_ERROR);
    }
    BESTX_FCLOSE(fp);
    fp=NULL;

    /* Write datamemory */
    BX_TRY(BestXDataMemWrite(handle,offset,st.st_size,data));
    BestXFree(data);
    data=NULL;
  }
  BX_TRY_CATCH
  {
    if (data) BestXFree(data);
    if (fp) BESTX_FCLOSE(fp);
  }
  BX_ERRETURN(BX_TRY_RET);
}

bx_errtype EXPORT BestXDataMemWrite(
    bx_handletype handle,
    bx_int32 int_addr,           /* qword boundary */
    bx_int32 num_bytes,          /* qword boundary */
    bx_int8ptr data_ptr
)
{
  BX_DECLARE_FUNCNAME("BestXDataMemWrite [datamemwrite]");

  BX_TRY_VARS;

  char reason[256];
  const bx_generic_infotype *mygen;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI);
    
    /* Check QWord-Alignment */
    BX_FCT_PARAM_ALIGNMENT_CHK(int_addr,8);
    BX_FCT_PARAM_ALIGNMENT_CHK(num_bytes,8);

    BX_TRY_FCT_PARAM_NULL_POINTER(data_ptr);

    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_DATA_MEM, &mygen));

    BESTX_SPRINTF(reason, "greater than data memory (%ld)", mygen->depth);
    BX_TRY_FCT_PARAM_R(1, int_addr > mygen->depth, reason);

    BESTX_SPRINTF(reason, "exceeds data memory (%ld)", mygen->depth);
    BX_TRY_FCT_PARAM_R(2, int_addr + num_bytes > mygen->depth, reason);

    /* Min and Max registers determine the Column -range 
       to program 
    */
  
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    BX_TRY(BestXDirectRegWrite(handle,BX_REG_DMEM_MIN_REG,
                 sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_DMEM_MAX_REG,
                 sizeof(bx_int16),1));

    while (num_bytes > 0)
    {
      bx_int32 chunkSize = num_bytes;
      
      if (bx_handlearray[handle].port==BX_PORT_FASTHIF)
      {
        /* FHIF burst restriction */
        chunkSize = min (num_bytes, FHIF_BLOCKSIZE_MAX);
      }
      if (bx_handlearray[handle].port==BX_PORT_USB)
      {
        /* FHIF burst restriction */
        chunkSize = min (num_bytes, USB_BLOCKSIZE_MAX);
      }

      /* Set starting row (i.e. offset) from which to start programming */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_DMEM_ADDR_CTR,
         sizeof(bx_int32),int_addr));

      /* Write to data memory in DWs */
      BX_TRY(BestXDirectRegBlockWrite(handle,
              BX_REG_DMEM_DATA_REG,
              sizeof(bx_int32), /* regwidth: DWORDS */
              1, /* 1=autoincrement */
              data_ptr,
              sizeof(bx_int32), /* datatype, DWORDS */
              chunkSize & ~0x7UL));

      
      int_addr += (chunkSize & ~0x7UL);
      data_ptr += (chunkSize & ~0x7UL);
      num_bytes -= (chunkSize & ~0x7UL);
    }
    
    /* Switch back to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }


  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDataMemRead(
 *
 * Purpose  : read data from data memory
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDataMemRead(
  bx_handletype handle,
  bx_int32 int_addr,
  bx_int32 num_bytes,
  bx_int8ptr data_ptr
)
{
  BX_DECLARE_FUNCNAME("BestXDataMemRead [datamemread]");

  BX_TRY_VARS;

  char reason[256];
  const bx_generic_infotype *mygen;

  BX_TRY_BEGIN
  {
    bx_int32 value;
    
    /* parameter and range checking */
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI);
    
    BX_TRY_FCT_PARAM_NULL_POINTER(data_ptr);

    BX_FCT_PARAM_ALIGNMENT_CHK(int_addr, 4);
    BX_FCT_PARAM_ALIGNMENT_CHK(num_bytes, 4);

    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_DATA_MEM, &mygen));

    if (int_addr > mygen->depth)
    {
      BESTX_SPRINTF(reason, "greater than data memory (%ld)", mygen->depth);
      BX_TRY_FCT_PARAM_R(1, int_addr > mygen->depth, reason);
    }

    if (int_addr + num_bytes > mygen->depth)
    {
      BESTX_SPRINTF(reason, "exceeds data memory (%ld)", mygen->depth);
      BX_TRY_FCT_PARAM_R(2, int_addr + num_bytes > mygen->depth, reason);
    }

    /* Min and Max registers determine the Column -range 
       to program */
    
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle, 0 /* no soft force */));
    
    /* first, get anything which is on uneven QWord address */
    if (int_addr & 0x4)
    {
      BX_TRY(BestXDirectRegWrite( handle,
          BX_REG_DMEM_MIN_REG,
          sizeof(bx_int16),
          1UL ));
      
      BX_TRY(BestXDirectRegWrite( handle,
          BX_REG_DMEM_MAX_REG,
          sizeof(bx_int16),
          1UL ));

      /* Set starting row (i.e. offset) from which to start reading */
      BX_TRY(BestXDirectRegWrite( handle,
          BX_REG_DMEM_ADDR_CTR,
          sizeof(bx_int32),
          int_addr & ~0x7 ));
      
      BX_TRY(BestXDirectRegRead( handle,
         BX_REG_DMEM_DATA_REG,
         4UL,
         &value ));

      *((bx_int32ptr) data_ptr) = value;
      data_ptr += 4;
      num_bytes -= 4;
      int_addr += 4;
    }

    if (num_bytes > 4)
    {
      BX_TRY(BestXDirectRegWrite( handle,
          BX_REG_DMEM_MIN_REG,
          sizeof(bx_int16),
          0UL ));
      
      BX_TRY(BestXDirectRegWrite( handle,
          BX_REG_DMEM_MAX_REG,
          sizeof(bx_int16),
          1UL ));

      while (num_bytes > 4)
      {
        bx_int32 chunkSize = num_bytes;

        if (bx_handlearray[handle].port==BX_PORT_FASTHIF)
        {
          /* FHIF burst restriction */
          chunkSize = min (num_bytes, FHIF_BLOCKSIZE_MAX);
        }
        if (bx_handlearray[handle].port==BX_PORT_USB)
        {
          /* USB burst restriction */
          chunkSize = min (num_bytes, USB_BLOCKSIZE_MAX);
        }

        if ((bx_handlearray[handle].port==BX_PORT_RS232) && (GetOsVersionShort() == OS_WIN95))
        {
            /* Serial burst restriction for Win98 - otherwise bytes get lost */
            chunkSize = min(num_bytes, 4096);
        }


        /* Set starting row (i.e. offset) from which to start reading */
        BX_TRY(BestXDirectRegWrite( handle,
            BX_REG_DMEM_ADDR_CTR,
            sizeof(bx_int32),
            int_addr & ~0x7 ));
  
        /* Read from data memory in DWs */
        BX_TRY(BestXDirectRegBlockRead(handle,
               BX_REG_DMEM_DATA_REG,
               4UL, /* regwidth */
               1, /* 1=autoincrement */
               data_ptr,
               4UL, /* data type, data mem is dwords */
               chunkSize & ~0x7UL ));
  
        int_addr += (chunkSize & ~0x7UL);
        data_ptr += (chunkSize & ~0x7UL);
        num_bytes -= (chunkSize & ~0x7UL);
      }
    }

    if (num_bytes == 4)
    {
      BX_TRY(BestXDirectRegWrite( handle,
          BX_REG_DMEM_MIN_REG,
          sizeof(bx_int16),
          0UL ));
      
      BX_TRY(BestXDirectRegWrite( handle,
          BX_REG_DMEM_MAX_REG,
          sizeof(bx_int16),
          0UL ));

      /* Set starting row (i.e. offset) from which to start reading */
      BX_TRY(BestXDirectRegWrite( handle,
          BX_REG_DMEM_ADDR_CTR,
          sizeof(bx_int32),
          int_addr & ~0x7 ));
      
      BX_TRY(BestXDirectRegRead( handle,
         BX_REG_DMEM_DATA_REG,
         4UL,
         &value ));
      
      *((bx_int32ptr) data_ptr) = value;
    }
    
    /* Switch back to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle, 0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }


  BX_ERRETURN(BX_TRY_RET);

}


/* --------------------------------------------------------------------------
 * Fills data memory with a caller-defined pattern.  The meaning of
 * params value1,2,3,4 are dependent on the pattern as follows;
 * pattern          value1              value2
 * ==============   =================== =========================
 * BX_PATT_CONST     mem set to this     ignored
 * BX_PATT_TOGGLE    1st var             2nd var
 * BX_PATT_COUNTER   starting val.       0==count down, else count up
 * BX_PATT_WALKBIT0  ignored             ignored
 * BX_PATT_WALKBIT1  ignored             ignored
 * BX_PATT_RANDOM    seed                ignored
 * BX_PATT_GROUNDBOUNCE 1st var          2nd var   (64bit toggler !!)        
 * BX_PATT_CROSSTALK                     
 * -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXDataMemPatternFill(
    bx_handletype handle,
    bx_int32 int_addr,           /* MUST be quadword aligned */
    bx_int32 num_bytes,          /* same */
    bx_int32 value1,             /* meaning depends on pattern */
    bx_int32 value2,             /* meaning depends on pattern */
    bx_int32 value3,             /* meaning depends on pattern */
    bx_int32 value4,             /* meaning depends on pattern */
    bx_dmempatttype pattern      /* type of pattern to use */
)
{
  BX_DECLARE_FUNCNAME("BestXDataMemPatternFill [datamempatternfill]");

  BX_TRY_VARS;

  bx_int32ptr data=NULL;
  const bx_generic_infotype *GenericDataMemInfo;
  bx_int32 depth;
  char reason[256];
  bx_int32 i,j;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI); 
    
    /* Check 128bit-Alignment */
    BX_FCT_PARAM_ALIGNMENT_CHK(int_addr,16);
    BX_FCT_PARAM_ALIGNMENT_CHK(num_bytes,16);

    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_DATA_MEM, &GenericDataMemInfo));
    depth = GenericDataMemInfo->depth;

    BESTX_SPRINTF(reason, "greater than data memory (%ld)", depth);
    BX_TRY_FCT_PARAM_R(1, int_addr > depth, reason);

    BESTX_SPRINTF(reason, "exceeds data memory (%ld)", depth);
    BX_TRY_FCT_PARAM_R(2, int_addr + num_bytes > depth, reason);

    data=(bx_int32ptr) BestXMemMalloc(num_bytes);
    BX_TRY_FAIL(data?BX_E_OK:BX_E_HOST_MEM_FULL);

    switch (pattern)
    {
      case BX_PATT_CONST32:
        for (i=0;i<num_bytes/4;i++)
        {
          data[i]=value1; 
        }
        break;
      case BX_PATT_CONST64:
      case BX_PATT_TOGGLE32:
        for (i=0;i<num_bytes/4;i+=2)
        {
          data[i]=value1; 
          data[i+1]=value2; 
        }
        break;
      case BX_PATT_TOGGLE64:
        for (i=0;i<num_bytes/4;i+=4)
        {
          data[i+0]=value1; 
          data[i+1]=value2; 
          data[i+2]=value3; 
          data[i+3]=value4; 
        }
        break;
      case BX_PATT_GROUNDBOUNCE32:
        for (i=0;i<num_bytes/4;i+=2)
        {
          data[i+0]=0x00000000;
          data[i+1]=0xffffffff;
        }
        break;
      case BX_PATT_GROUNDBOUNCE64:
        /* Datamemory is 64 bit wide !! */
        for (i=0;i<num_bytes/4;i+=4)
        {
          data[i+0]=data[i+1]=0x00000000;
          data[i+2]=data[i+3]=0xffffffff;
        }
        break;
      case BX_PATT_CROSSTALK32:
        /* Datamemory is 64 bit wide !! */
        for (i=0,j=0;i<num_bytes/4;i+=4,j=(j+1)%32)
        {
          data[i+0]=0;
          data[i+1]=~(1UL<<j);    /* only bit j remains unset */

          data[i+2]=0xffffffff;
          data[i+3]=1UL<<j;       /* only bit j remains set   */
        }
        break;
      case BX_PATT_CROSSTALK64:
        /* Datamemory is 64 bit wide !! */
        for (i=0,j=0;i<num_bytes/4;i+=8,j=(j+1)%64)
        {
          if (j<32)
          {
            data[i+0]=0;
            data[i+1]=0;
            data[i+2]=~(1UL<<j);    /* only bit j remains unset */
            data[i+3]=0xffffffff;

            data[i+4]=0xffffffff;
            data[i+5]=0xffffffff;
            data[i+6]=1UL<<j;    /* only bit j remains set */
            data[i+7]=0;
          }
          else
          {
            /* 32<=j<=63 */
            data[i+0]=0;
            data[i+1]=0;
            data[i+2]=0xffffffff;
            data[i+3]=~(1UL<<(j-32));    /* only bit j remains unset */

            data[i+4]=0xffffffff;
            data[i+5]=0xffffffff;
            data[i+6]=0;
            data[i+7]=1UL<<(j-32);    /* only bit j remains set */
          }
        }
        break;
      case BX_PATT_COUNTER32:
        if (value2)
        {
          /* going up */         
          for (i=0;i<num_bytes/4;i++)
          {
            data[i]= value1+i*4;
          }
        }
        else
        {
          /* going down */
          for (i=0;i<num_bytes/4;i++)
          {
            data[i]=value1-i*4;
          }
        }
        break;
      case BX_PATT_COUNTER64:
        /* !!! Not yet correct implemented !!!
           (no under/overflow allowed: upper 32bit fix on value2)
         */
        if (value3)
        {
          /* going up */         
          for (i=0;i<num_bytes/4;i+=2)
          {
            data[i]=value1+i*4;
            data[i+1]=value2;
          }
        }
        else
        {
          /* going down */
          for (i=0;i<num_bytes/4;i+=2)
          {
            data[i]=value1-i*4;
            data[i+1]=value2;
          }
        }
        break;
      case BX_PATT_WALKBIT032:
        for (i=0;i<num_bytes/4;i++)
        {
          data[i]= ~((bx_int32)(1<<(i%32)));
        }
        break;
      case BX_PATT_WALKBIT064:
        for (i=0,j=0;i<num_bytes/4;i+=2,j=(j+1)%64)
        {
          if (j<32)
          {
            data[i]   = ~((bx_int32)(1<<j));
            data[i+1] = 0xffffffff;
          }
          else
          {
            /* 32<=j<=63 */
            data[i]   = 0xffffffff;
            data[i+1] = ~((bx_int32)(1<<(j-32)));
          }
        }
        break;
      case BX_PATT_WALKBIT132:
        for (i=0;i<num_bytes/4;i++)
        {
          data[i]= (bx_int32)(1<<(i%32));
        }
        break;
      case BX_PATT_WALKBIT164:
        for (i=0,j=0;i<num_bytes/4;i+=2,j=(j+1)%64)
        {
          if (j<32)
          {
            data[i]   = 1UL<<j;
            data[i+1] = 0;
          }
          else
          {
            /* 32<=j<=63 */
            data[i]     = 0;
            data[i+1]   = 1UL<<(j-32);
          }
        }
        break;
      case BX_PATT_RANDOM:
        srand(value1);
        for (i=0;i<num_bytes/4;i++)
        {
          /* Note: RAND_MAX is usually 0x7fff */
          data[i]= ((((bx_int32)rand())<<17) & 0xfffe0000) | /* Bit 17-31 */
                   ((((bx_int32)rand())<< 2) & 0x0001fffc) | /* Bit  2-16 */
                   ((((bx_int32)rand())<< 0) & 0x00000003);  /* Bit  1- 0 */
        }
        break;
      default:
        BX_E_ERROR_MSG_SET("BestXDataMemPatternFill: Invalid value for parameter pattern")
        BX_TRY_FAIL(BX_E_ERROR);
        break;
    }

    BX_TRY(BestXDataMemWrite(handle,int_addr,num_bytes,(bx_int8ptr)data));

    BestXMemFree((void **)&data);
  }
  BX_TRY_CATCH
  {
    if (data) 
    {
      BestXMemFree((void **)&data);
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*****************************************************************/
/************************* Production Special Hack ***************/
/*****************************************************************/

bx_errtype EXPORT BestXOpenProduction(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;
  BX_TRY_BEGIN
  {
    BX_TRY(BestXExerciserDefaultSet(handle));
    BX_TRY(BestXRIBehDefaultSet(handle,0));
    BX_TRY(BestXRIBehSet(handle,0,BX_RIBEH_DELAY,100));
    BX_TRY(BestXRIBehSet(handle,0,BX_RIBEH_STEPS,4));
    BX_TRY(BestXRIBehSet(handle,0,BX_RIBEH_QUEUE,BX_RIBEH_QUEUE_A));
    BX_TRY(BestXRIBehSet(handle,0, BX_RIBEH_REQ64, 0)); /* don't try 64 bit transfer */
    
  } /* BX_TRY_BEGIN */
 
  BX_ERRETURN(BX_TRY_RET);
}




bx_errtype EXPORT BestXDirectRegHostWrite(
  bx_handletype handle,
  bx_int32 DeviceNr,
  bx_int32 dir_addr,
  bx_int32 regsize,    
  bx_int32 reg_value
)
/*
This function uses the access_port to write data to a certain 
memory-location on the card.
Each call to 'BestXDirectRegWrite' is redirected to here.
*/

{
  bx_int32 size;
  bx_int32 address;
  bx_handletype handle_host;
  bx_int8  datamem8[16];
  bx_int32 datamem[4];
  bx_int32 busaddress;
  bx_int32 buscmd;
  bx_int32 intaddress;
  
  BX_TRY_VARS_NO_PROG;

 
  BX_TRY_BEGIN
  {
    DBG_OUT(("We're doing a HostWrite\n"));
    /* bits 2::1 in access_cmd register indicate the size */
    switch(regsize)
    {
      case 1:
        size=0x2;
        break;
      case 2:
        size=0x4;
        break;
      case 4:
        size=0x0;
        break;
      default:
        size=0x0;
    }

    handle_host=handle;
    assert(DeviceNr<5);
    BX_TRY(BestXExerciserProgMode(handle_host,0));
    
    /* prepare values */
    address=((dir_addr<<8) | (0x00) | (size));
    busaddress= (ACCESS_CMD | 0x1<<(16+DeviceNr) | (DeviceNr<<11)); /* starts at 0x54 */
    buscmd=BX_RIBLK_BUSCMD_CONFIG_WRITE;
    intaddress=4;
    
    /* Write the address and value into the datamemory on card */
    datamem[0]=0xbeafbeaf;
    datamem[1]=address;
    datamem[2]=reg_value;
    datamem[3]=0xdeaddead;
    BESTX_MEMCPY( (void *) datamem8, (void *) datamem, sizeof(datamem8) );
    BX_TRY(BestXDataMemWrite(handle_host,0,16, &datamem8[0] ));
    
    /* set up the exerciser */
    BX_TRY(BestXRIGenSet(handle_host,BX_RIGEN_NUMBLK,1));
    BX_TRY(BestXRIGenSet(handle_host, BX_RIGEN_NUMBEH,1));
    BX_TRY(BestXRIBehDefaultSet(handle_host, 0));

    /* set behaviorlines */
    BX_TRY(BestXRIBehSet(handle_host,0,BX_RIBEH_REQ64,0));
    BX_TRY(BestXRIBehSet(handle_host,0, BX_RIBEH_STEPS, 4));
    BX_TRY(BestXRIBehSet(handle_host,0,BX_RIBEH_DELAY,5000));

    BX_TRY(BestXRIBlockDefaultSet(handle_host, 0));
    BX_TRY(BestXRIBlockSet(handle_host, 0, BX_RIBLK_BUSADDR_LO, busaddress));
    BX_TRY(BestXRIBlockSet(handle_host, 0, BX_RIBLK_RESOURCE, BX_RIBLK_RESOURCE_DATAMEM));
    BX_TRY(BestXRIBlockSet(handle_host, 0, BX_RIBLK_BUSCMD, buscmd));
    BX_TRY(BestXRIBlockSet(handle_host, 0, BX_RIBLK_INTADDR, 4));
    BX_TRY(BestXRIBlockSet(handle_host, 0, BX_RIBLK_NUMBYTES, sizeof(bx_int32) + regsize));
    BX_TRY(BestXRIProg(handle_host));
    BX_TRY(BestXExerciserRunMode(handle_host,0));
    DBG_OUT(("We start the exerciser now\n"));
    DBG_OUT(("...send to busaddress 0x%lx: 1. address 0x%lx // 2. Value 0x%lx \n",
              busaddress,
              address,
              reg_value));
    BX_TRY(BestXExerciserRun(handle_host));

  } /*BX_TRY_END*/

  BX_ERRETURN(BX_TRY_RET);
} /* DirectRegHostWrite */


bx_errtype EXPORT BestXDirectRegHostRead(
  bx_handletype handle,
  bx_int32 DeviceNr,
  bx_int32 dir_addr,
  bx_int32 regsize,
  bx_int32 *reg_value
)
/*
This function uses the access_port to receive data from a certain 
memory-location on the card.
Each call to 'BestXDirectRegRead' is redirected to here.
*/

{
  bx_int32 size;
  bx_handletype handle_host;
  bx_int32ptr addressptr;
  bx_int32 address;
  bx_int32 datamem[2];
  bx_int8 datamem8[8];

  BX_TRY_VARS_NO_PROG;
 
  BX_TRY_BEGIN
  {
    /* bits 2::1 in access_cmd register indicate the size */
    switch(regsize)
    {
      case 1:
        size=0x2;
        break;
      case 2:
        size=0x4;
        break;
      case 4:
        size=0x0;
        break;
      default:
        size=0x000;
    }

    DBG_OUT(("We're doing a HostRead\n"));
    handle_host= handle;
    assert(DeviceNr<5);
    BX_TRY(BestXExerciserProgMode(handle_host,0)); /* ask Chris !!! */
    
    /* prepare address value (0x01 is read command) */
    address=((dir_addr<<8) | (0x01) | (size));
    addressptr=&address;
    
    /* Write the address and data into the datamemory on card */
    datamem[0]=0xbeafbeaf;
    datamem[1]=address;
    BESTX_MEMCPY( (void *) datamem8, (void *) datamem, sizeof(datamem8) );
    BX_TRY(BestXDataMemWrite(handle_host,0,8, &datamem8[0] ));

    /* set up the exerciser */
    BX_TRY(BestXRIGenSet(handle_host,BX_RIGEN_NUMBLK,2));
    BX_TRY(BestXRIGenSet(handle_host,BX_RIGEN_NUMBEH,1));
    BX_TRY(BestXRIBehDefaultSet(handle_host, 0));

    /* set behaviorlines */
    BX_TRY(BestXRIBehSet(handle_host,0,BX_RIBEH_REQ64,0));
    BX_TRY(BestXRIBehSet(handle_host,0,BX_RIBEH_STEPS, 4));
    BX_TRY(BestXRIBehSet(handle_host,0,BX_RIBEH_DELAY,5000));

    /* program first line */
    BX_TRY(BestXRIBlockDefaultSet(handle_host, 0));
    BX_TRY(BestXRIBlockSet(handle_host, 0, BX_RIBLK_BUSADDR_LO, (ACCESS_CMD | 0x1<<(16+DeviceNr) | (DeviceNr<<11) )));
    BX_TRY(BestXRIBlockSet(handle_host, 0, BX_RIBLK_RESOURCE, BX_RIBLK_RESOURCE_DATAMEM));
    BX_TRY(BestXRIBlockSet(handle_host, 0, BX_RIBLK_BUSCMD, BX_RIBLK_BUSCMD_CONFIG_WRITE));
    BX_TRY(BestXRIBlockSet(handle_host, 0, BX_RIBLK_INTADDR, 4));
    BX_TRY(BestXRIBlockSet(handle_host, 0, BX_RIBLK_NUMBYTES, sizeof(bx_int32)));
    
    /* program second line */
    BX_TRY(BestXRIBlockDefaultSet(handle_host, 1));
    BX_TRY(BestXRIBlockSet(handle_host, 1, BX_RIBLK_BUSADDR_LO, ( ACCESS_DATA | 0x1<<(16+DeviceNr) | (DeviceNr<<11) )));
    BX_TRY(BestXRIBlockSet(handle_host, 1, BX_RIBLK_RESOURCE, BX_RIBLK_RESOURCE_DATAMEM));
    BX_TRY(BestXRIBlockSet(handle_host, 1, BX_RIBLK_BUSCMD, BX_RIBLK_BUSCMD_CONFIG_READ));
    BX_TRY(BestXRIBlockSet(handle_host, 1, BX_RIBLK_INTADDR, 8));
    BX_TRY(BestXRIBlockSet(handle_host, 1, BX_RIBLK_NUMBYTES, sizeof(bx_int32)));
    
    BX_TRY(BestXRIProg(handle_host));
    BX_TRY(BestXExerciserRunMode(handle_host,0));
    BX_TRY(BestXExerciserRun(handle_host));

    /* read result from the datamemory */
    BX_TRY(BestXDataMemRead(handle_host,8,8, (bx_int8ptr) datamem8));
    BESTX_MEMCPY( (void *) datamem, (void *) datamem8, sizeof(datamem) );
    *reg_value=datamem[0];
    
    DBG_OUT(("Read reg_value=0x%lx from the address 0x%lx\n",
              *reg_value,
              address));
   
  } /*BX_TRY_END*/

  BX_ERRETURN(BX_TRY_RET);

} /* BestXDirectRegHostRead */

/*******************************************************************/

